<?php

namespace BrittainWynyard\CatalogSuperStyle\Block;

use BrittainWynyard\CatalogSuperStyle\Helper\Data as SuperStyleHelper;
use Psr\Log\LoggerInterface;

class Swatch extends \EightWire\SuperStyleSwatch\Block\Catalog\Product\View\Swatch
{

    /**
     * @var SuperStyleHelper
     */
    private $superStyleHelper;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * Optimized Swatch constructor.
     * @param \Magento\Catalog\Block\Product\Context $context
     * @param \Magento\Framework\Stdlib\ArrayUtils $arrayUtils
     * @param \Magento\Catalog\Api\ProductRepositoryInterface $productRepository
     * @param \Magento\Framework\Api\SearchCriteriaBuilder $searchCriteriaBuilder
     * @param \Magento\Catalog\Model\Product\Gallery\ReadHandler $galleryReadHandler
     * @param \Magento\Catalog\Helper\Image $imageHelper
     * @param SuperStyleHelper $superStyleHelper
     * @param LoggerInterface $logger
     * @param array $data
     */
    public function __construct(
        \Magento\Catalog\Block\Product\Context $context,
        \Magento\Framework\Stdlib\ArrayUtils $arrayUtils,
        \Magento\Catalog\Api\ProductRepositoryInterface $productRepository,
        \Magento\Framework\Api\SearchCriteriaBuilder $searchCriteriaBuilder,
        \Magento\Catalog\Model\Product\Gallery\ReadHandler $galleryReadHandler,
        \Magento\Catalog\Helper\Image $imageHelper,
        SuperStyleHelper $superStyleHelper,
        LoggerInterface $logger,
        array $data = []
    ) {
        // Store our additional dependencies
        $this->superStyleHelper = $superStyleHelper;
        $this->logger = $logger;

        // Call parent constructor with required parameters
        parent::__construct(
            $context,
            $arrayUtils,
            $productRepository,
            $searchCriteriaBuilder,
            $galleryReadHandler,
            $imageHelper,
            $data
        );
    }



    /**
     * Get related swatch products using optimized index
     *
     * @return array
     */
    public function getRelatedProducts()
    {
        $product = $this->getProduct();
        if (!$product) {
            return [];
        }

        $superStyleId = $product->getSuperStyleId();
        $sku = $product->getSku();

        if (!$superStyleId) {
            return [];
        }

        // Use simple caching like parent class
        if (empty($this->relatedProducts)) {

            // Use our optimized index to get product IDs first
            $productIdsByStyleId = $this->superStyleHelper->getProductIdsBySuperstyleIds([$superStyleId]);
            $productIds = $productIdsByStyleId[$superStyleId] ?? [];

            if (empty($productIds)) {
                $this->relatedProducts = [];
                return [];
            }

            // Load full product objects only for the IDs we found
            $this->relatedProducts = $this->loadProductsByIds($productIds);
        }

        return $this->relatedProducts;
    }

    /**
     * Load products by IDs efficiently with performance tracking
     *
     * @param array $productIds
     * @return array
     */
    private function loadProductsByIds(array $productIds): array
    {
        if (empty($productIds)) {
            return [];
        }

        try {
            // Create fresh search criteria builder to avoid filter accumulation
            $searchCriteriaBuilder = clone $this->searchCriteriaBuilder;
            $searchCriteriaBuilder->addFilter('entity_id', $productIds, 'in');
            $searchCriteria = $searchCriteriaBuilder->create();

            $products = $this->productRepository->getList($searchCriteria)->getItems();
            $foundCount = count($products);
            $missingIds = array_diff($productIds, array_keys($products));

            if (!empty($missingIds)) {
                $this->logger->warning('SuperStyleSwatch: Some products not found', [
                    'requested_count' => count($productIds),
                    'found_count' => $foundCount,
                    'missing_count' => count($missingIds)
                ]);
            }

            // Load gallery data
            foreach ($products as $product) {
                $this->galleryReadHandler->execute($product);
            }

            return $products;

        } catch (\Exception $e) {
            $this->logger->error('SuperStyleSwatch: Failed to load products by IDs', [
                'error' => $e->getMessage(),
                'product_count' => count($productIds)
            ]);
            return [];
        }
    }

    // All other methods are inherited from parent EightWire\SuperStyleSwatch\Block\Catalog\Product\View\Swatch
}
