<?php

namespace Overdose\Base\Plugin\Checkout\Controller\Sidebar;

use Magento\Framework\App\Action\Context;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Serialize\SerializerInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\Checkout\Model\Session;
use Magento\Quote\Api\CartRepositoryInterface;
use Psr\Log\LoggerInterface;

class UpdateItemQtyPlugin
{
    /**
     * @var Session
     */
    private $checkoutSession;

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var \Magento\Framework\App\ViewInterface
     */
    private $view;

    /**
     * @var \Magento\Quote\Api\CartRepositoryInterface
     */
    private $cartRepository;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * UpdateItemQtyPlugin constructor.
     *
     * @param Context $context
     * @param CartRepositoryInterface $cartRepository
     * @param Session $checkoutSession
     * @param SerializerInterface $serializer
     * @param LoggerInterface $logger
     */
    public function __construct(
        Context $context,
        CartRepositoryInterface $cartRepository,
        Session $checkoutSession,
        SerializerInterface $serializer,
        LoggerInterface $logger
    ) {
        $this->view = $context->getView();
        $this->cartRepository  = $cartRepository;
        $this->checkoutSession = $checkoutSession;
        $this->serializer = $serializer;
        $this->logger = $logger;
    }

    /**
     * After execute get form from layout
     *
     * @param $subject
     * @param $result
     *
     * @return mixed
     */
    public function afterExecute($subject, $result)
    {
        $reloadSections = $subject->getRequest()->getParam('reload_sections');
        if ($reloadSections) {
            $content = $this->serializer->unserialize($result->getJsonData());
            try {
                if ($content['success']) {
                    $content = $this->fillSectionsData($reloadSections, $content);
                }
            } catch (\Exception $e) {
                $this->logger->critical($e);
                $content['success'] = false;
                $content['error_message'] = __('Something went wrong. Please refresh the page and try again.');
            }
            $result->setJsonData($this->serializer->serialize($content));
        }
        return $result;
    }

    /**
     * Insert sections data
     *
     * @param array $reloadSections
     * @param array $content
     *
     * @return array
     */
    private function fillSectionsData(array $reloadSections, array $content): array
    {
        if (!$this->getItemsQty()) {
            $content['reload_page'] = true;
        } else {
            $this->view->loadLayout('checkout_cart_index');
            $content['sections'] = [];

            foreach ($reloadSections as $sectionName) {
                $content['sections'][$sectionName] = $this->view->getLayout()->getBlock($sectionName)->toHtml();
            }
        }
        return $content;
    }

    /**
     * Get items QTY in cart
     *
     * @return int
     */
    private function getItemsQty(): int
    {
        try {
            $quote = $this->checkoutSession->getQuote();
            return (int)$quote->getItemsSummaryQty();
        } catch (NoSuchEntityException | LocalizedException $e) {
            $this->logger->critical($e);
            return 0;
        }
    }
}
