<?php

namespace Overdose\DataLayer\Helper;

use Magento\Customer\Api\Data\CustomerInterface;
use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\Exception\InputException;
use Magento\Framework\File\Csv;
use Magento\Framework\Filesystem\DirectoryList;
use Magento\Framework\Json\Helper\Data;
use Magento\Framework\Stdlib\Cookie\CookieMetadataFactory;
use Magento\Framework\Stdlib\Cookie\CookieSizeLimitReachedException;
use Magento\Framework\Stdlib\Cookie\FailureToSendException;
use Magento\Framework\Stdlib\CookieManagerInterface;
use Exception;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Customer\Api\AccountManagementInterface;
use Magento\Framework\App\Helper\Context;
use Magento\Sales\Model\Order;

/**
 * Class Cookies
 * @package Overdose\DataLayer\Helper
 */
class Cookies extends AbstractHelper
{

    /**
     * @var DirectoryList
     */
    protected DirectoryList $moduleReader;

    /**
     * @var Csv
     */
    protected Csv $fileCsv;

    /**
     * @var AccountManagementInterface
     */
    protected AccountManagementInterface $accountManagement;

    protected CookieManagerInterface $cookieManager;
    protected CookieMetadataFactory $cookieMetadataFactory;
    protected Data $jsonHelper;
    protected CustomerRepositoryInterface $customerRepository;
    protected OrderRepositoryInterface $orderRepository;
    protected SearchCriteriaBuilder $searchCriteriaBuilder;
    protected $orderModel;

    protected $countryPhoneMapping = [];

    /**
     * @param CookieManagerInterface $cookieManager
     * @param CookieMetadataFactory $cookieMetadataFactory
     * @param Data $jsonHelper
     * @param CustomerRepositoryInterface $customerRepository
     * @param OrderRepositoryInterface $orderRepository
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param AccountManagementInterface $accountManagement
     * @param DirectoryList $moduleReader
     * @param Csv $fileCsv
     */
    public function __construct(
        Context $context,
        CookieManagerInterface      $cookieManager,
        CookieMetadataFactory       $cookieMetadataFactory,
        Data                        $jsonHelper,
        CustomerRepositoryInterface $customerRepository,
        OrderRepositoryInterface    $orderRepository,
        SearchCriteriaBuilder       $searchCriteriaBuilder,
        AccountManagementInterface  $accountManagement,
        DirectoryList               $moduleReader,
        Csv                         $fileCsv,
        Order $orderModel
    ) {
        $this->cookieManager = $cookieManager;
        $this->cookieMetadataFactory = $cookieMetadataFactory;
        $this->jsonHelper = $jsonHelper;
        $this->customerRepository = $customerRepository;
        $this->orderRepository = $orderRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->accountManagement = $accountManagement;
        $this->moduleReader = $moduleReader;
        $this->fileCsv = $fileCsv;
        $this->orderModel = $orderModel;
        parent::__construct($context);
    }

    /**
     * @param string $event
     * @param $identification_location
     * @throws CookieSizeLimitReachedException
     * @throws FailureToSendException
     * @throws InputException
     */
    public function setDataLayerCookie(string $event, $identification_location)
    {
        $publicCookieMetadata = $this->cookieMetadataFactory->createPublicCookieMetadata()
            ->setDuration(3600)
            ->setPath('/')
            ->setHttpOnly(false);

        if ($event) {
            $customerData['event'] =  $event;
        }

        if ($identification_location) {
            $customerData['identification_location'] =  $identification_location;
        }

        if ($event) {
            $this->cookieManager->setPublicCookie('dataLayerEvent',
                $this->jsonHelper->jsonEncode($customerData),
                $publicCookieMetadata
            );
        }
    }

    /**
     * @param $email
     * @throws CookieSizeLimitReachedException
     * @throws FailureToSendException
     * @throws InputException
     */
    public function setDataLayerCookieUserIdentify($email, $order = null, $isOrder = false)
    {
        if ($email) {
            $customerData = [
                'event' => 'userIdentify',
                'email' => $email,
                'em' => hash('sha256', $email),
                'customerStatus' => 'not_customer',
                'order_count' => 0
            ];
            $customer = $this->getCustomerByEmail($email);
            $shippingAddress = null;

            if ($customer && !$order) {
                $customerData['customerId'] = $customer->getId();
                $customerData['first_name'] = mb_strtolower($customer->getFirstname());
                $customerData['fn'] = hash('sha256', mb_strtolower($customer->getFirstname()));
                $customerData['last_name'] = mb_strtolower($customer->getLastname());
                $customerData['ln'] = hash('sha256', mb_strtolower($customer->getLastname()));
                $shippingAddress = $this->accountManagement->getDefaultShippingAddress($customerData['customerId']);
            } elseif ($order && $isOrder) {
                $orderObj = $this->orderRepository->get($order->getId());
                $shippingAddress = $orderObj->getShippingAddress();
                $customerData['first_name'] = mb_strtolower($orderObj->getCustomerFirstname());
                $customerData['fn'] = hash('sha256', mb_strtolower($orderObj->getCustomerFirstname()));
                $customerData['last_name'] = mb_strtolower($orderObj->getCustomerLastname());
                $customerData['ln'] = hash('sha256', mb_strtolower($orderObj->getCustomerLastname()));
            }

            try {
                $customerData['country_code'] = 'nz';
                $customerData['country'] = hash('sha256', 'nz');
                if ($shippingAddress) {
                    if ($shippingAddress->getCity()) {
                        $city = $this->convertAddressString($shippingAddress->getCity());
                        $customerData['city'] = $city;
                        $customerData['ct'] = hash('sha256', $city);
                    }
                    if ($shippingAddress->getPostcode()) {
                        $zip = $this->convertAddressString($shippingAddress->getPostcode());
                        $customerData['zip'] = $zip;
                        $customerData['zp'] = hash('sha256', $zip);
                    }
                    if ($shippingAddress->getTelephone()) {
                        $phone = $this->getPhonePrefixByCountry($shippingAddress->getCountryId()) . ltrim($this->convertAddressString($shippingAddress->getTelephone()), "0");
                        $customerData['phone'] = $phone;
                        $customerData['ph'] = hash('sha256', $phone);
                    }
                    if (isset($shippingAddress->getStreet()[0]) && $shippingAddress->getStreet()[0]) {
                        $customerData['address'] = $shippingAddress->getStreet()[0];
                        $customerData['ad'] = hash('sha256', $shippingAddress->getStreet()[0]);
                    }
                }
            } catch (\Exception $e){}

            $orders = $this->getOrdersByEmail($email);
            if ($orders) {
                $customerData['order_count'] = count($orders);
                $customerData['customerStatus'] = 'customer';
                $lastOrder = end($orders);
                $customerData['order_amount'] = number_format((float)$lastOrder->getTotalPaid(), 2);
            }

            $publicCookieMetadata = $this->cookieMetadataFactory->createPublicCookieMetadata()
                ->setDuration(3600)
                ->setPath('/')
                ->setHttpOnly(false);
            $this->cookieManager->setPublicCookie('tmpDataLayer',
                $this->jsonHelper->jsonEncode($customerData),
                $publicCookieMetadata
            );
        }
    }

    /**
     * @param $address
     * @return string
     */
    private function convertAddressString($address)
    {
        return str_replace(array("?", " ", ",", ";"), "", mb_strtolower($address));
    }

    /**
     * @param string $email
     * @return CustomerInterface|string
     */
    protected function getCustomerByEmail(string $email)
    {
        try {
            return $this->customerRepository->get($email);
        } catch (Exception $e) {
            return '';
        }
    }

    public function getOrderById($id)
    {
        return $this->orderModel->loadByIncrementId($id);
    }

    /**
     * @param $email
     * @return array
     */
    protected function getOrdersByEmail($email): array
    {
        $searchCriteria = $this->searchCriteriaBuilder->addFilter('customer_email', $email)
            ->create();

        return $this->orderRepository->getList($searchCriteria)->getItems();
    }

    /**
     * @param $countryCode
     * @return array|mixed|string|string[]|null
     * @throws Exception
     */
    protected function getPhonePrefixByCountry($countryCode)
    {
        if (!count($this->countryPhoneMapping)) {
            $file = $this->moduleReader->getRoot() . '/app/code/Overdose/DataLayer/Data/country_phone_prefix.csv';
            $this->fileCsv->setDelimiter(',');
            $data = $this->fileCsv->getData($file);
            foreach ($data as $countryMapping) {
                $this->countryPhoneMapping[strtolower($countryMapping[2])] = preg_replace('/[^0-9]/', '', $countryMapping[1]);
            }
        }
        if (isset($this->countryPhoneMapping[mb_strtolower($countryCode)])) {
            return $this->countryPhoneMapping[mb_strtolower($countryCode)];
        }
        return '';
    }
}
