<?php

namespace BrittainWynyard\Searchspring\Model;

use Magento\Catalog\Model\Product\Attribute\Source\Status;
use Magento\Catalog\Model\Product\Visibility;
use Magento\Catalog\Model\ResourceModel\Product\Collection;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory;
use Magento\Store\Model\StoreManagerInterface;
use BrittainWynyard\CatalogSuperStyle\Model\ResourceModel\Index\CollectionFactory as SuperStyleIndexCollectionFactory;
use Psr\Log\LoggerInterface;

class SearchSpringStyles
{
    public const STYLE_ATTRIBUTE = 'super_style_id';
    
    /**
     * Constructor
     *
     * @param CollectionFactory $productCollectionFactory
     * @param Visibility $visibility
     * @param StoreManagerInterface $storeManager
     * @param SuperStyleIndexCollectionFactory $superStyleIndexCollectionFactory
     * @param LoggerInterface $logger
     */
    public function __construct(
        private readonly CollectionFactory $productCollectionFactory,
        private readonly Visibility        $visibility,
        private readonly StoreManagerInterface $storeManager,
        private readonly SuperStyleIndexCollectionFactory $superStyleIndexCollectionFactory,
        private readonly LoggerInterface $logger,
    ) {
    }

    /**
     * Get products by super style IDs using the optimized index
     *
     * @param array $styleIDs
     * @return Collection
     */
    public function getProducts(array $styleIDs): Collection
    {
        if (empty($styleIDs)) {
            return $this->productCollectionFactory->create();
        }

        // Get current website ID
        $websiteId = (int) $this->storeManager->getWebsite()->getId();
        
        // Get product IDs from the super style index
        $productIds = $this->getProductIdsFromIndex($styleIDs, $websiteId);
        
        if (empty($productIds)) {
            return $this->productCollectionFactory->create();
        }

        // Create collection with the product IDs from index
        $collection = $this->productCollectionFactory->create();
        $collection
            ->addAttributeToSelect(['name', 'image', 'url_key', 'sku', 'super_style_id'])
            ->addFieldToFilter('entity_id', ['in' => $productIds])
            ->addAttributeToFilter('status', Status::STATUS_ENABLED)
            ->addAttributeToFilter('visibility', $this->visibility->getVisibleInSiteIds());
            
        return $collection;
    }

    /**
     * Get product IDs from the super style index using the collection
     *
     * @param array $styleIDs
     * @param int $websiteId
     * @return array
     */
    private function getProductIdsFromIndex(array $styleIDs, int $websiteId): array
    {
        $superStyleIndexCollection = $this->superStyleIndexCollectionFactory->create();
        
        $productIds = $superStyleIndexCollection->getAllProductIdsByStyleIds($styleIDs, $websiteId);
        
        return $productIds;
    }

}
