<?php

declare(strict_types=1);

namespace Overdose\InventoryShipping\Ui\DataProvider;

use Magento\InventoryApi\Api\Data\SourceInterface;
use Magento\InventoryApi\Api\GetSourcesAssignedToStockOrderedByPriorityInterface;
use Magento\InventoryApi\Api\SourceRepositoryInterface;
use Magento\InventoryShippingAdminUi\Ui\DataProvider\GetSourcesByOrderIdSkuAndQty as GetSourcesByOrderIdSkuAndQtyDef;
use Magento\InventorySourceSelectionApi\Api\Data\ItemRequestInterfaceFactory;
use Magento\InventorySourceSelectionApi\Api\GetDefaultSourceSelectionAlgorithmCodeInterface;
use Magento\InventorySourceSelectionApi\Api\SourceSelectionServiceInterface;
use Magento\InventorySourceSelectionApi\Model\GetInventoryRequestFromOrder;
use Overdose\InventoryShipping\Model\ResourceModel\OrderSource\GetOrderSourceCodesByOrderId;

class GetSourcesByOrderIdSkuAndQty extends GetSourcesByOrderIdSkuAndQtyDef
{
    private GetSourcesAssignedToStockOrderedByPriorityInterface $getSourcesAssignedToStockOrderedByPriority;

    private ItemRequestInterfaceFactory $itemRequestFactory;

    private GetInventoryRequestFromOrder $getInventoryRequestFromOrder;

    private SourceRepositoryInterface $sourceRepository;

    /**
     * @var SourceInterface[]
     */
    private array $sources;
    private GetOrderSourceCodesByOrderId $getOrderSourceCodesByOrderId;

    /**
     * @param ItemRequestInterfaceFactory $itemRequestFactory
     * @param SourceSelectionServiceInterface $sourceSelectionService
     * @param GetDefaultSourceSelectionAlgorithmCodeInterface $getDefaultSourceSelectionAlgorithmCode
     * @param GetInventoryRequestFromOrder $getInventoryRequestFromOrder
     * @param SourceRepositoryInterface $sourceRepository
     * @param GetSourcesAssignedToStockOrderedByPriorityInterface $getSourcesAssignedToStockOrderedByPriority
     * @param GetOrderSourceCodesByOrderId $getOrderSourceCodesByOrderId
     */
    public function __construct(
        ItemRequestInterfaceFactory $itemRequestFactory,
        SourceSelectionServiceInterface $sourceSelectionService,
        GetDefaultSourceSelectionAlgorithmCodeInterface $getDefaultSourceSelectionAlgorithmCode,
        GetInventoryRequestFromOrder $getInventoryRequestFromOrder,
        SourceRepositoryInterface $sourceRepository,
        GetSourcesAssignedToStockOrderedByPriorityInterface $getSourcesAssignedToStockOrderedByPriority,
        GetOrderSourceCodesByOrderId $getOrderSourceCodesByOrderId
    ) {
        $this->getSourcesAssignedToStockOrderedByPriority = $getSourcesAssignedToStockOrderedByPriority;
        $this->getInventoryRequestFromOrder = $getInventoryRequestFromOrder;
        $this->itemRequestFactory = $itemRequestFactory;
        $this->sourceRepository = $sourceRepository;
        $this->getOrderSourceCodesByOrderId = $getOrderSourceCodesByOrderId;

        parent::__construct(
            $itemRequestFactory,
            $sourceSelectionService,
            $getDefaultSourceSelectionAlgorithmCode,
            $getInventoryRequestFromOrder,
            $sourceRepository
        );
    }

    /**
     * Get sources by order id sku and qty
     *
     * @param int $orderId
     * @param string $sku
     * @param float $qty
     * @return array
     */
    public function execute(int $orderId, string $sku, float $qty): array
    {
        $result = [];
        $orderSourceCodes = $this->getOrderSourceCodesByOrderId->execute($orderId);
        if (!empty($orderSourceCodes) && isset($orderSourceCodes[$sku])) {
            foreach ($orderSourceCodes[$sku] as $orderSources) {
                $sourceCode = $orderSources['sourceCode'];
                $result[] = [
                    'sourceName' => $this->getSourceName($sourceCode),
                    'sourceCode' => $sourceCode,
                    'qtyAvailable' => $orderSources[GetOrderSourceCodesByOrderId::QUANTITY],
                    'qtyToDeduct' => $orderSources[GetOrderSourceCodesByOrderId::QUANTITY]
                ];
            }
        }
        return $result;
    }

    private function getSourceName(string $sourceCode): string
    {
        if (!isset($this->sources[$sourceCode])) {
            $this->sources[$sourceCode] = $this->sourceRepository->get($sourceCode)->getName();
        }

        return $this->sources[$sourceCode];
    }
}
